<?php

class Aroos_REST_Video_Controller extends Aroos_REST_Controller
{
    /**
     * Endpoint namespace
     *
     * @var string
     */
    protected $namespace = 'aroos/v1';

    /**
     * Route name
     *
     * @var string
     */
    protected $base = 'videos';

    /**
     * Post type
     *
     * @var string
     */
    protected $post_type = 'video';

    /**
     * Post status
     * 
     * @var array
     */
    protected $post_status = array('publish', 'pending', 'draft');

    /**
     * Post custom fields
     * 
     * @var array
     */
    protected $post_attributes = array('video_id');

    /**
     * Register all routes releated with stores
     *
     * @return void
     */
    public function register_routes()
    {
        register_rest_route($this->namespace, '/' . $this->base, array(
            'args' => array(
                'id' => array(
                    'description' => __('Unique identifier for the object.', 'aroos-api'),
                    'type' => 'integer',
                ),
            ),
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_items'),
                'args' => $this->get_collection_params(),
                'permission_callback' => array($this, 'get_video_permissions_check'),
            ),
        ));

        register_rest_route($this->namespace, '/' . $this->base . '/(?P<id>[\d]+)/', array(
            'args' => array(
                'id' => array(
                    'description' => __('Unique identifier for the object.', 'aroos-api'),
                    'type' => 'integer',
                ),
            ),
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_item'),
                'args' => $this->get_collection_params(),
                'permission_callback' => array($this, 'get_video_permissions_check'),
            ),
        ));
    }

    /**
     * Get video object
     * 
     * @param integer $id
     * @return object
     */
    public function get_object($id)
    {
        if (! get_post($id)) {
            return new WP_Error('aroos_rest_invalid_' . $this->post_type . '_id', sprintf(__('Ivalid ID', 'aroos-api'), __METHOD__), array( 'status' => '404' ));
        }

        return get_post($id);
    }

    /**
     * Get video permissions check
     * 
     * @return bool
     */
    public function get_video_permissions_check()
    {
        return true;
    }

    /**
     * Get video data
     * 
     * @param $video
     * @param $request
     * @return array
     */
    protected function prepare_data_for_response($video, $request)
    {
        $content = apply_filters('the_content', $video->post_content);

        $data = array(
            'id' => $video->ID,
            'title' => $video->post_title,
            'content' => get_media_embedded_in_content($content)[0],
            'thumbnail' => $this->add_additional_data_when(get_the_post_thumbnail_url($video->ID), get_the_post_thumbnail_url($video->ID)),
            'permalink' => get_post_permalink($video->ID),
            'date_created' => $video->post_date,
            'date_created_gmt' => $video->post_date_gmt,
            'date_modified' => $video->post_modified,
            'date_modified_gmt' => $video->post_modified_gmt,
            'attributes' => $this->prepare_data_attributes($video),
        );

        $response = rest_ensure_response($data);
        $response->add_links($this->prepare_links($video, $request));

        return $response;
    }

    /**
     * Prepare links for the request.
     *
     * @param WP_Post         $object  Object data.
     * @param WP_REST_Request $request Request object.
     *
     * @return array                   Links for the given post.
     */
    protected function prepare_links($object, $request)
    {
        $links = array(
            'self'       => array(
                'href' => rest_url( sprintf( '/%s/%s/%d', $this->namespace, $this->base, $object->ID ) ),
            ),
            'collection' => array(
                'href' => rest_url( sprintf( '/%s/%s', $this->namespace, $this->base ) ),
            ),
        );

        return $links;
    }
}