<?php

class Aroos_REST_Photo_Controller extends Aroos_REST_Controller
{
    /**
     * Endpoint namespace
     *
     * @var string
     */
    protected $namespace = 'aroos/v1';

    /**
     * Route name
     *
     * @var string
     */
    protected $base = 'photos';

    /**
     * Post type
     *
     * @var string
     */
    protected $post_type = 'photo';

    /**
     * Post status
     * 
     * @var array
     */
    protected $post_status = array('publish', 'pending', 'draft');

    /**
     * Post custom fields
     * 
     * @var array
     */
    protected $post_attributes = array('image');

    /**
     * Register all routes releated with stores
     *
     * @return void
     */
    public function register_routes()
    {
        register_rest_route($this->namespace, '/' . $this->base, array(
            'args' => array(
                'id' => array(
                    'description' => __('Unique identifier for the object.', 'aroos-api'),
                    'type' => 'integer',
                ),
            ),
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_items'),
                'args' => $this->get_collection_params(),
                'permission_callback' => array($this, 'get_photo_permissions_check'),
            ),
        ));
    }

    /**
     * Get photo object
     * 
     * @param integer $id
     * @return object
     */
    public function get_object($id)
    {
        if (! get_post($id)) {
            return new WP_Error('aroos_rest_invalid_' . $this->post_type . '_id', sprintf(__('Ivalid ID', 'aroos-api'), __METHOD__), array( 'status' => '404' ));
        }

        return get_post($id);
    }

    /**
     * Get photo permissions check
     * 
     * @return bool
     */
    public function get_photo_permissions_check()
    {
        return true;
    }

    /**
     * Get photo data
     * 
     * @param $photo
     * @param $request
     * @return array
     */
    protected function prepare_data_for_response($photo, $request)
    {
        $data = array(
            'id' => $photo->ID,
            'permalink' => get_post_permalink($photo->ID),
            'date_created' => $photo->post_date,
            'date_created_gmt' => $photo->post_date_gmt,
            'date_modified' => $photo->post_modified,
            'date_modified_gmt' => $photo->post_modified_gmt,
            'attributes' => $this->prepare_data_attributes($photo),
        );

        $response = rest_ensure_response($data);
        $response->add_links($this->prepare_links($photo, $request));

        return $response;
    }

    /**
     * Prepare links for the request.
     *
     * @param WP_Post         $object  Object data.
     * @param WP_REST_Request $request Request object.
     *
     * @return array                   Links for the given post.
     */
    protected function prepare_links($object, $request)
    {
        $links = array(
            'self'       => array(
                'href' => rest_url( sprintf( '/%s/%s/%d', $this->namespace, $this->base, $object->ID ) ),
            ),
            'collection' => array(
                'href' => rest_url( sprintf( '/%s/%s', $this->namespace, $this->base ) ),
            ),
        );

        return $links;
    }
}