<?php

class Aroos_REST_Order_Controller extends Aroos_REST_Controller
{
    /**
     * Endpoint namespace
     *
     * @var string
     */
    protected $namespace = 'aroos/v1';

    /**
     * Route name
     *
     * @var string
     */
    protected $base = 'orders';

    /**
     * Post type
     *
     * @var string
     */
    protected $post_type = 'all_orders';

    /**
     * Post status
     * 
     * @var array
     */
    protected $post_status = array('publish', 'pending', 'draft');

    /**
     * Post custom fields
     * 
     * @var array
     */
    protected $post_attributes = array('fullname', 'phone', 'country', 'kind', 'date', 'the_order', 'ordernum');

    /**
     * Register all routes releated with stores
     *
     * @return void
     */
    public function register_routes()
    {
        register_rest_route($this->namespace, '/' . $this->base, array(
            'args' => array(
                'id' => array(
                    'description' => __('Unique identifier for the object.', 'aroos-api'),
                    'type' => 'integer',
                ),
            ),
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_items'),
                'args' => $this->get_collection_params(),
                'permission_callback' => array($this, 'get_order_permissions_check'),
            ),
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'create_item'),
                'args' => $this->get_endpoint_args_for_item_schema(WP_REST_Server::CREATABLE),
                'schema' => array($this, 'get_public_item_schema'),
                'permission_callback' => array($this, 'create_order_permissions_check'),
            ),
        ));   
        
        register_rest_route($this->namespace, '/' . $this->base . '/(?P<id>[\d]+)/', array(
            'args' => array(
                'id' => array(
                    'description' => __('Unique identifier for the object.', 'aroos-api'),
                    'type' => 'integer',
                ),
            ),
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'get_item'),
                'args' => $this->get_collection_params(),
                'permission_callback' => array($this, 'get_order_permissions_check'),
            ),
        ));
    }

    /**
     * Get order object
     * 
     * @param integer $id
     * @return object
     */
    public function get_object($id)
    {
        if (! get_post($id)) {
            return new WP_Error('aroos_rest_invalid_' . $this->post_type . '_id', sprintf(__('Ivalid ID', 'aroos-api'), __METHOD__), array( 'status' => '404' ));
        }

        return get_post($id);
    }

    /**
     * Get order permissions check
     * 
     * @return bool
     */
    public function get_order_permissions_check()
    {
        return true;
    }

    /**
     * Create order permissions check
     * 
     * @return bool
     */
    public function create_order_permissions_check()
    {
        return true;
    }
    
    /**
     * Get order data
     * 
     * @param $order
     * @param $request
     * @return array
     */
    protected function prepare_data_for_response($order, $request)
    {
        $data = array(
            'id' => $order->ID,
            'title' => $order->post_title,
            'date_created' => $order->post_date,
            'date_created_gmt' => $order->post_date_gmt,
            'date_modified' => $order->post_modified,
            'date_modified_gmt' => $order->post_modified_gmt,
            'attributes' => $this->prepare_data_attributes($order),
            'status' => $this->get_taxonomy_terms($order, 'order-statue'),
        );

        $response = rest_ensure_response($data);
        $response->add_links($this->prepare_links($order, $request));

        return $response;
    }

    /**
     * Prepare object for database mapping
     * 
     * @param object $request
     * @param bool $creating
     * @return object
     */
    protected function prepare_object_for_database($request, $creating = false)
    {
        $request = json_decode($request->get_body(), true);
        
        $form_data = array(
            'post_title' => $request['post_title'],
            'post_status' => 'publish',
            'post_type' => $this->post_type,
        );

        $post = wp_insert_post($form_data);
        
        if (is_wp_error($post)) {
            return $post;
        }
        
        $post = get_post($post);

        foreach ($this->post_attributes as $attribute) {
            if ($attribute !== 'ordernum') {
                update_field($attribute, $request[$attribute], $post->ID);
            } else {
                update_field('ordernum', $post->ID, $post->ID);
            }
            
            if($attribute === 'the_order') {
                foreach ($request[$attribute] as $song) {
                    update_post_meta($song['id'], 'orders_num', (int) get_post_meta($song['id'], 'orders_num', 0) + 1);
                }
            }
        }

        wp_set_object_terms($post->ID, array(5), 'order-statue');

        return $post;
    }

    /**
     * Get taxonomy terms.
     *
     * @param WP_Post $order
     * @param string $taxonomy
     * @return array
     */
    protected function get_taxonomy_terms($order, $taxonomy = 'cat')
    {
        $terms = array();

        foreach (wp_get_post_terms($order->ID, $taxonomy, array('fields' => 'all')) as $term) {
            $terms[] = array(
                'id' => $term->term_id,
                'name' => $term->name,
            );
        }

        return $terms;
    }

    /**
     * Prepare links for the request.
     *
     * @param WP_Post         $object  Object data.
     * @param WP_REST_Request $request Request object.
     *
     * @return array                   Links for the given post.
     */
    protected function prepare_links($object, $request)
    {
        $links = array(
            'self'       => array(
                'href' => rest_url( sprintf( '/%s/%s/%d', $this->namespace, $this->base, $object->ID ) ),
            ),
            'collection' => array(
                'href' => rest_url( sprintf( '/%s/%s', $this->namespace, $this->base ) ),
            ),
        );

        return $links;
    }
}