<?php

/**
 * Base REST Controller
 * 
 * @since 1.0.0
 * @package aroos-api
 */
abstract class Aroos_REST_Controller extends WP_REST_Controller
{
    /**
     * Get object
     * 
     * @param int $id
     * @return WC_Data|WP_Error
     */
    protected function get_object($id)
    {
        return new WP_Error('invalid-method', sprintf(__("Method '%s' not implemented. Must be overridden in subclass.", 'aroos-api'), __METHOD__), array('status' => '405'));
    }

    /**
     * Get a collection of posts
     * 
     * @param WP_REST_Request $request
     * @return WP_Error|WP_REST_Response
     */
    public function get_items($request)
    {
        $query_args = $this->prepare_objects_query($request);
        $query = new WP_Query();
        $result = $query->query($query_args);
        
        if (count($result) <= 0) {
            return new WP_Error('aroos_rest_invalid_' . $this->post_type . '_id', sprintf(__('No data found', 'aroos-api'), __METHOD__), array( 'status' => '404' ));
        }

        $data = array();
        $objects = array_map(array($this, 'get_object'), $result);
        $data_objects = array();

        foreach ($objects as $object) {
            $data = $this->prepare_data_for_response($object, $request);

            $data_objects[] = $this->prepare_response_for_collection($data);
        }

        $response = rest_ensure_response($data_objects);
        $response = $this->format_collection_response($response, $request, $query->found_posts);

        return $response;
    }

    /**
     * Get item for an object
     * 
     * @param $request
     * @return object
     */
    public function get_item($request)
    {
        $id = (int) $request['id'];
        $post = get_post($id);

        if (empty($id) || empty($post->ID) || $post->post_type != $this->post_type) {
            return new WP_Error('aroos_api_rest_invalid_' . $this->post_type . '_id', __('Ivalid ID', 'aroos-api'), array('status' => '404'));
        }

        $data = $this->prepare_data_for_response($this->get_object($post), $request);
        $response = rest_ensure_response($data);

        return $response;
    }

    /**
     * Create an item
     * 
     * @param $request
     * @return void
     */
    public function create_item($request)
    {
        $validate = $this->validation_before_create_item($request);

        if (is_wp_error($validate)) {
            return $validate;
        }

        $object = $this->prepare_object_for_database($request, true);
            
        if (is_wp_error($object)) {
            return $object;
        }

        if ($object->ID) {
            return $this->prepare_data_for_response($this->get_object($object->ID), $request);
        }

        return new WP_Error('invalid-creating-request', __('Invalid creating request', 'aroos-api'), array('status' => '500'));
    }

    /**
     * Update an item
     * 
     * @param $request
     * @return void
     */
    public function update_item($request)
    {
        $validate = $this->validation_before_update_item($request);

        if (is_wp_error($validate)) {
            return $validate;
        }

        $object = $this->prepare_object_for_database($request, false);

        if (is_wp_error($object)) {
            return $object;
        }

        if ($object->status === true && $object->id) {
            return $this->prepare_data_for_response($this->get_object($object->id), $request);
        }

        return new WP_Error('invalid-updating-request', __('Invalid updating request', 'aroos-api'), array('status' => '500'));
    }

    /**
     * Update an item
     * 
     * @param $request
     * @return void
     */
    public function delete_item($request)
    {
        $object = $this->get_object((int) $request['id']);

        if (is_wp_error($object)) {
            return $object;
        }

        $data = $this->prepare_data_for_response($object, $request);
        $response = rest_ensure_response($data);

        $object->delete(true);
        $result = 0 === $object->get_id();

        if (!$result) {
            return new WP_Error('aroos_api_rest_cannot_delete', sprintf(__('The %s cannot be deleted.', 'aroos-api'), $this->post_type), array('status' => '500'));
        }

        do_action('aroos_api_rest_delete_' . $this->post_type . '_object', $object, $response, $request);
        
        return $response;
    }

    /**
     * Prepare_object_for_database
     *
     * @return WP_Error
     */
    protected function prepare_object_for_database( $request ) {
        return new WP_Error( 'invalid-method', sprintf( __( "Method '%s' not implemented. Must be overridden in subclass.", 'aroos-api' ), __METHOD__ ), array( 'status' => '405' ) );
    }

    /**
     * Prepares a response for insertion into a collection.
     *
     * @param WP_REST_Response $response Response object.
     * @return array|mixed Response data, ready for insertion into collection data.
     */
    public function prepare_response_for_collection( $response ) {
        if ( ! ( $response instanceof WP_REST_Response ) ) {
            return $response;
        }

        $data   = (array) $response->get_data();
        $server = rest_get_server();

        if (method_exists($server, 'get_compact_response_links')) {
            $links = call_user_func(array($server, 'get_compact_response_links'), $response);
        } else {
            $links = call_user_func(array($server, 'get_response_links'), $response);
        }

        if (!empty($links)) {
            $data['_links'] = $links;
        }

        return $data;
    }

    /**
     * Validate before create an Item
     *
     * @return bool|WP_Error
     */
    protected function validation_before_create_item( $request ) {
        return true;
    }

    /**
     * Validate before update an Item
     *
     * @return bool|WP_Error
     */
    protected function validation_before_update_item( $request ) {
        return true;
    }

    /**
     * Prepares the object for the REST response.
     *
     * @since  1.0.0
     * @param  $object  Object data.
     * @param  WP_REST_Request $request Request object.
     * @return WP_Error|WP_REST_Response Response object on success, or WP_Error object on failure.
     */
    protected function prepare_data_for_response( $object, $request ) {
        return new WP_Error( 'invalid-method', sprintf( __( "Method '%s' not implemented. Must be overridden in subclass.", 'aroos-api' ), __METHOD__ ), array( 'status' => '405' ) );
    }

    /**
     * Prepare objects query.
     * 
     * @param WP_REST_Request $request
     * @return array
     */
    protected function prepare_objects_query($request)
    {
        $args = array();

        $args['fields'] = 'ids';
        $args['post_status'] = !isset($request['post_status']) ? $this->post_status : $request['post_status'];
        $args['offset'] = $request['offset'];
        $args['order'] = $request['order'];
        $args['orderby'] = $request['orderby'];
        $args['paged'] = $request['page'];
        
        if (!empty($request['include'])) {
            $args['post__in'] = explode(',', $request['include']);
        }
        
        if (!empty($request['exclude'])) {
            $args['post__not_in'] = explode(',', $request['exclude']);
        }
        $args['posts_per_page'] = !empty($request['per_page']) ? $request['per_page'] : 10;
        
        if (!empty($request['slug'])) {
            $args['name'] = $request['slug'];
        }
        
        if (!empty($request['parent'])) {
            $args['post_parent__in'] = explode(',', $request['parent']);
        }
        
        if (!empty($request['parent_exclude'])) {
            $args['post_parent__not_in'] = explode(',', $request['parent_exclude']);
        }
        
        if (!empty($request['s'])) {
            $args['s'] = $request['s'];
        }
        
        $args['meta_key'] = $request['meta_key'];
        $args['meta_value'] = $request['meta_value'];
        
        if ('date' === $args['orderby']) {
            $args['orderby'] = 'date ID';
        }

        if (!isset($args['orderby'])) {
            $args['order_by'] = 'post_date';
        }

        $args['date_query'] = array();

        if (isset($request['before'])) {
            $args['date_query'][0]['before'] = $request['before'];
        }

        if (isset($request['after'])) {
            $args['date_query'][0]['after'] = $request['after'];
        }
        
        if (isset($request['tax_value']) && isset($request['tax_name'])) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => $request['tax_name'],
                    'field' => 'id',
                    'terms' => array($request['tax_value']),
                ),
            );
        }

        $args['post_type'] = $this->post_type;

        $args = apply_filters('aroos_api_rest_' . $this->post_type . '_object_query', $args, $request);

        return $this->prepare_items_query($args, $request);
    }

    /**
     * Determine the allowed query_vars for a get_items() response and
     * prepare for WP_Query.
     *
     * @param array           $prepared_args
     * @param WP_REST_Request $request
     * @return array          $query_args
     */
    public function prepare_items_query($parepared_args = array(), $request = null)
    {
        $valid_vars = array_flip($this->get_allowed_query_vars());

        $query_args = array();
        foreach ($valid_vars as $var => $index) {
            if (isset($parepared_args[$var])) {
                $query_args[$var] = apply_filters('aroos_api_rest_query_var-' . $var, $parepared_args[$var]);
            }
        }

        $query_args['ignore_sticky_posts'] = true;

        if (isset($query_args['orderby']) && 'include' === $query_args['orderby']) {
            $query_args['orderby'] = 'post__in';
        } elseif (isset($query_args['orderby']) && 'id' === $query_args['orderby']) {
            $query_args['orderby'] = 'ID';
        }

        return $query_args;
    }

    /**
     * Get custom attributes of the post
     * 
     * @param WP_Post $object
     * @return array
     */
    public function prepare_data_attributes($object)
    {
        if (!isset($this->post_attributes) || !is_array($this->post_attributes)) {
            return array();
        }
        
        $attributes = [];

        foreach ($this->post_attributes as $attribute) {
            $value = get_field($attribute, $object->ID);
            
            if (is_integer($value)) {
                if (get_attached_file($attachment_id = get_post($value)->ID)) {
                    $attachment_data = wp_get_attachment_metadata($value);
                    $value = array(
                        'format' => $attachment_data['fileformat'],
                        'mime_type' => $attachment_data['mime_type'],
                        'size' => $attachment_data['filesize'],
                        'url' => wp_get_attachment_url($attachment_id),
                    );

                    if ($attachment_data['mime_type'] == 'audio/mpeg') {
                        $mp3file = new MP3FILE(get_attached_file($attachment_id));
                        $value['duration'] = MP3FILE::formatTime($mp3file->getDurationEstimate());
                    }
                }
            }
            
            $attributes[$attribute] = $value;
        }

        return $attributes;
    }

    /**
     * Get all the WP Query vars that are allowed for the API request.
     *
     * @return array
     */
    protected function get_allowed_query_vars()
    {
        global $wp;

        $valid_vars = apply_filters('query_vars', $wp->public_query_vars);
        $post_type_obj = get_post_type_object($this->post_type);

        $rest_valid = array(
            'date_query',
            'ignore_sticky_posts',
            'offset',
            'post_status',
            'post__in',
            'post__not_in',
            'post_parent',
            'post_parent__in',
            'post_parent__not_in',
            'posts_per_page',
            'meta_query',
            'tax_query',
            'meta_key',
            'meta_value',
            'meta_compare',
            'meta_value_num',
            's',
        );

        $valid_vars = array_merge($valid_vars, $rest_valid);
        $valid_vars = apply_filters('aroos_api_rest_query_vars', $valid_vars);

        return $valid_vars;
    }

    /**
     * Format item's collection for response
     *
     * @param  object $response
     * @param  object $request
     * @param  array $items
     * @param  int $total_items
     *
     * @return object
     */
    public function format_collection_response($response, $request, $total_items)
    {
        if ($total_items === 0) {
            return $response;
        }

        $per_page = (int) (!empty($request['per_page']) ? $request['per_page'] : 20);
        $page = (int) (!empty($request['page']) ? $request['page'] : 1);

        $response->header('X-WP-Total', (int) $total_items);
        $max_pages = ceil($total_items / $per_page);

        $response->header('X-WP-TotalPages', (int) $max_pages);
        $base = add_query_arg($request->get_query_params(), rest_url(sprintf('/%s/%s', $this->namespace, $this->base)));

        if ($page > 1) {
            $prev_page = $page - 1;

            if ($prev_page > $max_pages) {
                $prev_page = $max_pages;
            }

            $prev_link = add_query_arg('page', $prev_page, $base);
            $response->link_header('prev', $prev_link);
        }

        if ($max_pages > $page) {
            $next_page = $page + 1;
            $next_link = add_query_arg('page', $next_page, $base);
            $response->link_header('next', $next_link);
        }
        
        $data = $response->get_data();
        
        $response->set_data(array(
            'data' => $data,
            'links' => array(
                'prev' => isset($prev_link) ? $prev_link : null,
                'next' => isset($next_link) ? $next_link : null,
            ),
        ));

        return $response;
    }

    /**
     * Retrieve a value based on a given condition.
     * 
     * @param bool $condition
     * @param mixed $value
     * @param mixed $default 
     */
    protected function add_additional_data_when($condition, $value, $default = null)
    {
        if ($condition) {
            return $value;
        }

        return func_num_args() === 3 ? $default : '';
    }
}